<?php
/**
 * Tiny.cc REST API v3.1 client
 * 
 * This client works with tiny.cc URL shortening services using API v3.
 * 
 * Methods prefixed with 'mass_' use curl_multi to establish several parallel connections for faster processing.
 * On communication error these method DOESN'T throw exceptions. Instead they returne error codes for failed URLs.
 * But they still may throw exceptions if invalid arguments passed.
 * 
 * @copyright	2015 Tiny.cc
 * @author		Alexey Gorshunov <ag@blazing.pro>
 * @license		MIT
 * @package		tinycc/client
 */
 
namespace tinycc;

require_once "client_exception.php";

/**
 * Client class
 */
class client
{
	protected $api_root_url;
	protected $username;
	protected $api_key;
	protected $api_version = "3.1";
	protected $unknown_error = array("code"=>101, "message"=>'Unknown error','details'=>'');
	
	protected $selection = null;
	protected $working_domain = null;
	protected $long_url_duplicates = null;
	
/**
 * Client contructor
 * @param	array	$config		Required keys: 'api_root_url', 'username', 'api_key', Optional keys: 'batch_operations_limit', 'parallel_streams'
 * @api
 */	
	public function __construct(array $config)
	{
		$this->api_root_url = @$config['api_root_url'];
		$this->username = @$config['username'];
		$this->api_key = @$config['api_key'];
		$this->batch_operations_limit = isset($config['batch_operations_limit']) ? $config['batch_operations_limit'] : 30;
		$this->parallel_streams = isset($config['parallel_streams']) ? $config['parallel_streams'] : 4;
		
		if(empty($this->api_root_url)
			OR filter_var($this->api_root_url, FILTER_VALIDATE_URL) === false){
				throw client_exception::invalid_argument(__METHOD__,"api_root_url");
		}
		
		$this->api_root_url = rtrim($this->api_root_url,"/")."/";
		
		if(empty($this->username)){
			throw client_exception::invalid_argument(__METHOD__,"username");
		}

		if(empty($this->api_key)){
			throw client_exception::invalid_argument(__METHOD__,"api_key");
		}
	}
	
/**
 * Change working domain (if not used, all operations performed for default domain)
 * @param	string 	domain
 * @return	void
 */	
	public function set_working_domain($domain)
	{
		$this->working_domain = $domain;
	}
	
	
/**
 * https://tinycc.com/tiny/api-docs#shorten
 * @param	string 	long_url_duplicates
 * @return	void
 */	
	public function set_long_url_duplicates($long_url_duplicates)
	{
		$this->long_url_duplicates = $long_url_duplicates;
	}	
	
	
/**
 * Get account info
 * Resulting array contains 'counters' section with actual API usage and limits.
 * @api
 * @return	array
 */		
	public function account_info()
	{
		return $this->simple_api_call("GET","account");
	}	
	
/**
 * Shorten long URL
 * On communication error this method throws exception
 * @api
 * @param	string 	$long_url
 * @param	array	$data	optional URL properties. Possible keys: 'custom_hash', 'note', 'is_qrcode', 'favorite', 'email_stats', 'expiration_date'
 * @return	array	shortened URL properties
 */	
	public function shorten($long_url, array $data=array())
	{
		if(empty($long_url)
			OR filter_var($long_url, FILTER_VALIDATE_URL) === false){
				throw client_exception::invalid_argument(__METHOD__,"long_url");
		}

		$data['long_url'] = $long_url;
		$query = array();
		
		if($this->long_url_duplicates){
			$query['disable_long_url_duplicates'] = $this->long_url_duplicates;
		}
		
		$result = $this->simple_api_call("POST","urls",$query,array(
			'urls'=>array($data)
		));
		
		if(empty($result['urls'][0])){
			throw client_exception::client_error("Empty response");
		}
		
		$url = $result['urls'][0];
		
		if(!empty($url['error']['code'])){
			throw client_exception::api_error($url['error']['message'], $url['error']['code']); 
		}
		
		unset($url['error']);
		
		return $url;
	}

/**
 * Shorten many long URLs at once
 * @api
 * @param	array 	$long_urls each element of array is a string representing long URL
 * @param	array	$data	optional URL properties. Possible keys: 'custom_hash', 'note', 'is_qrcode', 'favorite', 'email_stats', 'expiration_date'
 * @return	array	shortened URLs properties
 */	
	public function mass_shorten(array $long_urls, array $data=array())
	{
		if(empty($long_urls)){
			throw client_exception::invalid_argument(__METHOD__,"long_urls");
		}

		if(!is_array($data)){
			throw client_exception::invalid_argument(__METHOD__,"data");
		}

		$self = $this;
		$results = $this->parallel_processing($long_urls, function($portion)use($self, $data){
			$urls = array();
			foreach($portion as $long_url):
				$url = $data;
				$url['long_url'] = $long_url;
				$urls[] = $url;
			endforeach;
			$query = array();
			
			if($self->long_url_duplicates){
				$query['disable_long_url_duplicates'] = $self->long_url_duplicates;
			}			
			return $self->prepare_curl("POST","urls",$query,array("urls"=>$urls));
		},"long_url");		
		return $results;
	}
	
/**
 * Read index of short URLs
 * @api
 * @param	array 	$params pagination options. Possible keys: 'offset', 'limit', 'search' (search by string), 
 *				 		'order_by' (order by one of following: "created", "favorite", "tagged", "qr", "clicks", "hash")
 * @return	array	page of shortened URLs
 */		
	public function read_page(array $params = array())
	{
		$params = array_merge(array(
			'limit'=>$this->batch_operations_limit,
			'offset'=>0,
			'search'=>'',
			'order_by'=>'created'
		),$params);
		
		$start_time = microtime(true);
		$calls = $params['limit'] / $this->batch_operations_limit;
		$urls = array();
		$results_count = 0;
		for($i=0;$i<$calls;$i++){
			$portion = $this->simple_api_call("GET","urls", array_merge($params,
						array(
							'offset'=>$params['offset']+$i*$this->batch_operations_limit,
							'limit'=>min($this->batch_operations_limit, $params['limit']-$i*$this->batch_operations_limit)
						)));
			$total_count = $portion['page']['total_count'];
			$results_count += $portion['page']['results_count'];
			$urls = array_merge($urls, $portion['urls']);
		}
		
		$results = array(
			'urls' => $urls,
			'page' => array(
				'results_count' => $results_count,
				'total_count' => $total_count,
				'offset' => $params['offset']
			),
			'meta'=>array(
				'request_time' => (microtime(true) - $start_time)
			)
		);

		return $results;
	}	
	
	
	public function select_with_hashes(array $hashes)
	{
		$this->selection = array(
			'type'=>'hashes',
			'arg'=>$hashes
		);
		return $this;
	}

	public function select_with_tags($tags_query)
	{
		if(!is_string($tags_query)){
			throw client_exception::invalid_argument(__METHOD__,"tags_query");
		}		
		
		$this->selection = array(
			'type'=>'tags',
			'arg'=>$tags_query
		);
		return $this;
	}

/**
 * Read short URLs properties
 * On communication error this method may throw exception
 * @api
 * @return	array	shortened URL properties
 */	
	public function read(array $params = array())
	{
		if(empty($this->selection) OR empty($this->selection['arg'])){
			throw client_exception::missing_selection(__METHOD__);
		}

		if($this->selection['type'] == 'hashes'){
			if(count($this->selection['arg']) == 1){
				return $this->simple_api_call("GET","urls",array('hashes'=>$this->selection['arg']));
			}else{
				$self = $this;
				$results = $this->parallel_processing($hashes, function($portion)use($self){
					return $self->prepare_curl("GET","urls",array("hashes"=>implode(",",$portion)));
				},"hash");		
				return $results;
			}
		}elseif($this->selection['type'] == 'tags'){
			$params['tags'] = $this->selection['arg'];
			return $this->read_page($params);
		}
	}
	


/**
 * Change properties of short URL(s)
 * On communication error this method may throw exception
 * @api
 * @param	array	$data	new properties. It supports the same keys as shorten method
 * @return	array	shortened URL properties
 */
	public function edit(array $data)
	{
		if(empty($this->selection) OR empty($this->selection['arg'])){
			throw client_exception::missing_selection(__METHOD__);
		}

		if(empty($data) OR !is_array($data)){
			throw client_exception::invalid_argument(__METHOD__,"data");
		}
		
		if($this->selection['type'] == 'hashes'){
			if(count($this->selection['arg']) == 1){
				return $this->simple_api_call("PATCH","urls",array('hashes'=>$this->selection['arg']),array('urls'=>array($data)));
			}else{
				$self = $this;
				$results = $this->parallel_processing($this->selection['arg'], function($portion)use($self, $data){
					$urls = array();
					foreach($portion as $hash):
						$url = $data;
						$url['hash'] = $hash;
						$urls[] = $url;
					endforeach;
					return $self->prepare_curl("PATCH","urls",array(),array("urls"=>$urls));
				},"hash");		
				return $results;
			}
		}elseif($this->selection['type'] == 'tags'){
			return $this->simple_api_call("PATCH","urls",array('tags'=>$this->selection['arg']),array("urls"=>array($data)));
		}		
	}
	
/**
 * Delete short URL(s)
 * This method ignores not existing URLs
 * On communication error this method may throw exception
 * @api
 * @return	void
 */	
	public function delete()
	{
		if(empty($this->selection) OR empty($this->selection['arg'])){
			throw client_exception::missing_selection(__METHOD__);
		}		
		
		if($this->selection['type'] == 'hashes'){
			if(count($this->selection['arg']) == 1){
				$this->simple_api_call("DELETE","urls",array('hashes'=>$this->selection['arg']));
			}else{
				$self = $this;
				$this->parallel_processing($hashes, function($portion)use($self){
					return $self->prepare_curl("DELETE","urls",array("hashes"=>implode(",",$portion)));
				});		
			}
		}elseif($this->selection['type'] == 'tags'){
			$this->simple_api_call("DELETE","urls",array('tags'=>$this->selection['arg']));
		}
	}


	


/**
 * Reset clicks stats for short URL(s)
 * This method ignores not existing URLs
 * On communication error this method may throw exception
 * @api
 * @return	void
 */	
	public function reset_stats()
	{
		if(empty($this->selection) OR empty($this->selection['arg'])){
			throw client_exception::missing_selection(__METHOD__);
		}		

		if($this->selection['type'] == 'hashes'){
			if(count($this->selection['arg']) == 1){
				$this->simple_api_call("DELETE","stats",array('hashes'=>$this->selection['arg']));
			}else{
				$self = $this;
				$this->parallel_processing($hashes, function($portion)use($self){
					return $self->prepare_curl("DELETE","stats",array("hashes"=>implode(",",$portion)));
				});		
			}
		}elseif($this->selection['type'] == 'tags'){
			$this->simple_api_call("DELETE","stats",array('tags'=>$this->selection['arg']));
		}
	}
	
	
/**
 * Get list of available tags
 * @api
 * @return	array
 */		
	public function tags()
	{
		return $this->simple_api_call("GET","tags");
	}	

/**
 * Get list of available domains
 * @api
 * @return	array
 */		
	public function domains()
	{
		return $this->simple_api_call("GET","domains");
	}	

/**
 * Create new tag (if not exists)
 * @api
 * @return	array
 */		
	public function create_tag($label)
	{
		return $this->simple_api_call("POST","tags",array(),array('label'=>$label));
	}
	
/**
 * Delete tag label
 * @api
 * @return	array
 */		
	public function delete_tag($label)
	{
		return $this->simple_api_call("DELETE","tags/".urlencode($label));
	}	
	
	
/**
 * @internal
 */	
	protected function parallel_processing(array $items, $callback, $sort_results_by=null)
	{
		if(empty($items))return false; 
		$start_time = microtime(true);
		$mh = curl_multi_init();
		$results = array();
		if($sort_results_by){
			$results['urls'] = array();
		}

		$curls = array();
		$portions = array_chunk($items, $this->batch_operations_limit);
		for($portion_index = 0; $portion_index < count($portions);){
		
			$curls = array();
			for($i=0
				;$i < $this->parallel_streams 
					AND $portion_index < count($portions)
				;$i++, $portion_index++
				){
				$curls[$portion_index] = $curl = call_user_func($callback, $portions[$portion_index]);
				curl_multi_add_handle($mh,$curl);
			}
			
			do {
				curl_multi_exec($mh,$running);
			} while($running > 0);

			foreach($curls as $index=>$curl):
				if($sort_results_by){
					$results['urls'] = array_merge($results['urls'], 
						$this->sort_results($portions[$index], $curl, $sort_results_by)
					);
				}
				curl_multi_remove_handle($mh, $curl);
			endforeach;
		};

		curl_multi_close($mh);	
		$results['meta'] = array('request_time' => (microtime(true) - $start_time));
		return $results;
	}
	
/**
 * @internal
 */	
	protected function sort_results($portion, $curl, $sort_results_by)
	{
		$items = array();
		$response_code=curl_getinfo($curl, CURLINFO_HTTP_CODE);
		$response = curl_multi_getcontent($curl);
		$data=json_decode($response,true);
		
		if(!empty($data['error']['code'])
			OR empty($data) 
			OR !isset($data['urls'])
			OR $response_code != '200'
			){
			if(!empty($data['error']['code'])){
				$error = $data['error'];
			}elseif($response_code != '200'){
				$error = array("code"=>1308, "message"=>'Connection error','details'=>"HTTP code {$response_code}");
			}else{
				$error = $this->unknown_error;
			}
			
			foreach($portion as $item):
				$items[] = array(
					$sort_results_by => $item,
					"error"=>$error
				);
			endforeach;
			
		}else{
			$this->check_version($data['version']);
			$items = $data['urls'];
		}
		
		return $items;
	}
	
/**
 * @internal
 */	
	protected function prepare_curl($method, $resource, $get_params = array(), $post_data = null)
	{
		if($this->working_domain){
			$get_params['domain'] = $this->working_domain;
		}
		$query_string = empty($get_params) ? "" : ("?".http_build_query($get_params));
		
		$curl = curl_init(); //cUrl Init
		curl_setopt($curl, CURLOPT_RETURNTRANSFER, 1);
		curl_setopt($curl, CURLOPT_HEADER, 0); // show headers
		curl_setopt($curl, CURLOPT_FOLLOWLOCATION, 1);
		curl_setopt($curl, CURLOPT_HTTPHEADER, array(
			'Accept: application/json',
			'Content-type: application/json',
			"Cache-control: no-cache",
			"Authorization: Basic ".base64_encode($this->username.":".$this->api_key)
		));
		curl_setopt($curl, CURLOPT_URL, $this->api_root_url.$resource.$query_string);
		curl_setopt($curl, CURLOPT_CUSTOMREQUEST, strtoupper($method));
		if($post_data){
			curl_setopt($curl, CURLOPT_POST, 1); 
			curl_setopt($curl, CURLOPT_POSTFIELDS, json_encode($post_data));
		}
		
		return $curl;
	}
	
/**
 * @internal
 */	
	protected function simple_api_call($method, $resource, $get_params = array(), $post_data = null)
	{
		$curl = $this->prepare_curl($method, $resource, $get_params, $post_data);
		return $this->curl_request($curl);
	}
	
/**
 * @internal
 */	
	protected function curl_request($curl)
	{
		$start_time = microtime(true);
		
		$response = curl_exec($curl);
		$response_code = curl_getinfo($curl, CURLINFO_HTTP_CODE);
		
		if (!$response) {
			throw client_exception::curl_error($curl);
		}
		$data = json_decode($response, true);
		curl_close($curl);
		
		if($response_code != '200'){
			if(!empty($data['error']["code"])){
				throw client_exception::api_error($data['error']["message"], $data['error']["code"]);
			}else{
				throw client_exception::api_error($response);
			}
		}
		unset($data['error']);
		$this->check_version($data['version']);
		unset($data['version']);
		
		if(isset($data['meta']['time'])){
			$processing_time = $data['meta']['time'];
		}
		
		$data['meta'] = array('request_time' => (microtime(true) - $start_time));
		
		if(isset($processing_time)){
			$data['meta']['processing_time'] = $processing_time;
		}
		return $data;
	}
	
/**
 * @internal
 */	
	protected function check_version($version)
	{
		if($version != $this->api_version){
			throw client_exception::client_error("Version of client ({$this->api_version}) doesn't match version of API ({$version})");
		}
	}
	
}


