﻿using System.Net.Http.Headers;
using System.Text;

namespace Tinycc.Helpers;

/// <summary>
/// Extensions for setting up http client with default settings
/// </summary>
public static class ClientExtensions
{
    /// <summary>
    /// Setting BaseAddress, MediaType and UserAgent to default headers
    /// </summary>
    public static void SetDefaultSettings(this HttpClient client)
    {
        client.BaseAddress = new Uri(Constants.DefaultBaseAdress);

        // Accepted media type
        client.DefaultRequestHeaders.Accept.Clear();
        client.DefaultRequestHeaders.Accept.Add(new MediaTypeWithQualityHeaderValue(Constants.JsonMediaType));

        // User agent
        var libVersion = typeof(Client).Assembly.GetName().Version;
        client.DefaultRequestHeaders.UserAgent.ParseAdd($"Tinycc.NET/{libVersion}");
    }

    /// <summary>
    /// Setting headers with access details
    /// </summary>
    public static void UseHeadersAuthentication(this HttpClient client, string username, string apiKey)
    {
        if (username == null)
            throw new ArgumentNullException(nameof(username));

        if (apiKey == null)
            throw new ArgumentNullException(nameof(apiKey));

        client.DefaultRequestHeaders.Add("X-Tinycc-User", username);
        client.DefaultRequestHeaders.Add("X-Tinycc-Key", apiKey);
    }

    /// <summary>
    /// Setting basic authentication
    /// </summary>
    public static void UseBasicAuthentication(this HttpClient client, string username, string apiKey)
    {
        if (username == null)
            throw new ArgumentNullException(nameof(username));

        if (apiKey == null)
            throw new ArgumentNullException(nameof(apiKey));

        // Set basic authentification
        var byteArray = Encoding.UTF8.GetBytes(string.Format("{0}:{1}", username, apiKey));
        client.DefaultRequestHeaders.Authorization = new AuthenticationHeaderValue("Basic", Convert.ToBase64String(byteArray));
    }
}
