﻿using Newtonsoft.Json;
using Newtonsoft.Json.Converters;
using Newtonsoft.Json.Linq;
using Newtonsoft.Json.Serialization;
using Tinycc.Helpers;
using Tinycc.Resources;

namespace Tinycc;

/// <summary>
/// Tinycc API client
/// </summary>
public class Client : BaseClient, IClient
{
    public Client(
        string username, string apiKey, bool useBasicAuth = false, 
        int batchOperationsLimit = Constants.BatchOperationsLimit,
        int parallelStreams = Constants.ParallelStreams)
        : base(username, apiKey, useBasicAuth, batchOperationsLimit, parallelStreams)
    { }

    public async Task<AccountResponse> AccountInfoAsync()
    {
        var request = new Request(Account.Resource);

        Response clientResponse = await GetAsync(request).ConfigureAwait(false);

        return clientResponse.GetValue<AccountResponse>(AccountResponse.DataItem);
    }

    public async Task<DomainsResponse> DomainsAsync()
    {
        var request = new Request(Domains.Resource);

        Response clientResponse = await GetAsync(request).ConfigureAwait(false);

        return clientResponse.GetList<DomainsResponse>(DomainsResponse.DataItem);
    }

    public async Task<TagsResponse> TagsAsync()
    {
        var request = new Request(Tags.Resource);

        Response clientResponse = await GetAsync(request).ConfigureAwait(false);

        return clientResponse.GetList<TagsResponse>(TagsResponse.DataItem);
    }

    public async Task<CreateTagResponse> CreateTagAsync(string label)
    {
        var createTagParams = new CreateTagParams
        {
            Label = label
        };

        var request = new Request(CreateTag.Resource)
        {
            Body = JObject.FromObject(createTagParams)
        };

        Response clientResponse = await PostAsync(request).ConfigureAwait(false);

        return clientResponse.GetList<CreateTagResponse>(CreateTagResponse.DataItem);
    }

    public async Task<EditTagResponse> EditTagAsync(string label, string newLabel)
    {
        var editTagParams = new EditTagParams
        {
            Label = newLabel
        };

        var request = new Request(UrlHelper.CombineUrl(EditTag.Resource, label))
        {
            Body = JObject.FromObject(editTagParams)
        };

        Response clientResponse = await PatchAsync(request).ConfigureAwait(false);

        return clientResponse.GetList<EditTagResponse>(EditTagResponse.DataItem);
    }

    public async Task<DeleteTagResponse> DeleteTagAsync(string label)
    {
        var request = new Request(UrlHelper.CombineUrl(DeleteTag.Resource, label));

        Response clientResponse = await DeleteAsync(request).ConfigureAwait(false);

        return clientResponse.GetValue<DeleteTagResponse>(DeleteTagResponse.DataItem);
    }

    public async Task<ShortenResponse> ShortenAsync(string url, string disableLongUrlDuplicates = null)
    {
        return await ShortenAsync(new ShortenParams { LongUrl = url}, disableLongUrlDuplicates);
    }

    public async Task<ShortenResponse> ShortenAsync(ShortenParams url, string disableLongUrlDuplicates = null)
    {
        var urlParams = new MassShortenParams
        {
            Urls = [url]
        };

        var request = new Request(Shorten.Resource)
        {
            Body = JObject.FromObject(urlParams)
        };

        if (disableLongUrlDuplicates != null)
            request.Filter(Shorten.DisableLongUrlDuplicates, disableLongUrlDuplicates);

        Response clientResponse = await PostAsync(request).ConfigureAwait(false);

        var results = clientResponse.GetList<MassShortenResponse>(MassShortenResponse.DataItem);

        return results.Urls.First();
    }

    public async Task<MassShortenResponse> MassShortenAsync(IEnumerable<ShortenParams> urls, string disableLongUrlDuplicates = null)
    {
        var responses = await MassProcessAsync(urls, (chunk) =>
        {
            var urlParams = new MassShortenParams
            {
                Urls = chunk
            };

            var request = new Request(Shorten.Resource)
            {
                Body = JObject.FromObject(urlParams)
            };

            if (disableLongUrlDuplicates != null)
                request.Filter(Shorten.DisableLongUrlDuplicates, disableLongUrlDuplicates);

            return PostAsync(request);
        });

        return new MassShortenResponse
        {
            Urls = responses.SelectMany(r => r.GetList<MassShortenResponse>(MassShortenResponse.DataItem).Urls)
        };
    }

    public async Task<ReadResponse> ReadPageAsync(
        string search = null, IEnumerable<string> hashes = null, IEnumerable<string> tags = null,
        int? offset = null, int? limit = null, ReadOrderBy? orderBy = null)
    {
        if (Domain != null && tags != null)
            throw new ArgumentException("Domain parameter doesn't work together with tags parameter");

        if (hashes != null && tags != null)
            throw new ArgumentException("Hashes parameter doesn't work together with tags parameter");

        var filterFunc = (Request request, IEnumerable<string> chunk) =>
        {
            if (search != null)
                request.Filter(Read.Search, search);

            if (chunk != null)
                request.Filter(Read.Hashes, string.Join(',', chunk));

            if (tags != null)
                request.Filter(Read.Tags, string.Join(',', tags));

            if (offset != null)
                request.Filter(Read.Offset, offset.Value);

            if (limit != null)
                request.Filter(Read.Limit, limit.Value);

            if (orderBy != null && orderBy != ReadOrderBy.Undefined)
                request.Filter(Read.OrderBy, orderBy.Value.GetEnumMemberValue());
        };

        if (hashes == null)
        {
            // Simple API call
            var request = new Request(Read.Resource);

            filterFunc(request, null);

            Response clientResponse = await GetAsync(request).ConfigureAwait(false);

            return clientResponse.GetList<ReadResponse>(ReadResponse.DataItem);
        }
        else
        {
            // Parallel API calls
            var clientResponses = await MassProcessAsync(hashes, (chunk) =>
            {
                var request = new Request(Read.Resource);

                filterFunc(request, chunk);

                return GetAsync(request);
            });

            var urls = clientResponses.SelectMany(r => r.GetList<ReadResponse>(ReadResponse.DataItem).Urls);
            if (orderBy != null && orderBy != ReadOrderBy.Undefined)
            {
                if (orderBy == ReadOrderBy.Clicks)
                    urls = urls.OrderBy(u => u.TotalClicks);
                else if (orderBy == ReadOrderBy.Created)
                    urls = urls.OrderBy(u => u.Created);
                else if (orderBy == ReadOrderBy.Hash)
                    urls = urls.OrderBy(u => u.Hash);
                else if (orderBy == ReadOrderBy.Modified)
                    urls = urls.OrderBy(u => u.Modified);
            }

            return new ReadResponse { Urls = urls };
        }
    }

    public async Task<ReadResponse> ReadAsync(
        string search = null, IEnumerable<string> hashes = null, IEnumerable<string> tags = null)
    {
        return await ReadPageAsync(search, hashes, tags).ConfigureAwait(false);
    }

    public async Task<EditResponse> EditAsync(EditParams url)
    {
        var urlParams = new MassEditParams
        {
            Urls = [url]
        };

        var request = new Request(UrlHelper.CombineUrl(Edit.Resource, url.Hash))
        {
            Body = JObject.FromObject(urlParams, new JsonSerializer { NullValueHandling = NullValueHandling.Ignore })
        };

        Response clientResponse = await PatchAsync(request).ConfigureAwait(false);

        var results = clientResponse.GetList<MassEditResponse>(MassEditResponse.DataItem);

        return results.Urls.First();
    }

    public async Task<MassEditResponse> EditAsync(IEnumerable<EditParams> urls)
    {
        var responses = await MassProcessAsync(urls, (chunk) =>
        {
            var urlParams = new MassEditParams
            {
                Urls = chunk
            };

            var request = new Request(Edit.Resource)
            {
                Body = JObject.FromObject(urlParams, new JsonSerializer { NullValueHandling = NullValueHandling.Ignore })
            };

            return PatchAsync(request);
        });

        return new MassEditResponse
        {
            Urls = responses.SelectMany(r => r.GetList<MassEditResponse>(MassEditResponse.DataItem).Urls)
        };
    }

    public async Task<MassEditResponse> EditAsync(EditParams data, IEnumerable<string> hashes = null, IEnumerable<string> tags = null)
    {
        if (hashes == null && tags == null)
            throw new ArgumentException("Please specify Hashes or Tags selector to edit urls");

        if (hashes != null && tags != null)
            throw new ArgumentException("Hashes parameter doesn't work together with tags parameter");

        if (hashes == null)
        {
            // Simple API call
            var urlParams = new MassEditParams
            {
                Urls = [data]
            };

            var request = new Request(Edit.Resource)
            {
                Body = JObject.FromObject(urlParams, new JsonSerializer { NullValueHandling = NullValueHandling.Ignore })
            };

            request.Filter(Edit.Tags, string.Join(',', tags));

            Response clientResponse = await PatchAsync(request).ConfigureAwait(false);

            return clientResponse.GetList<MassEditResponse>(MassEditResponse.DataItem);
        }
        else
        {
            // Parallel API calls
            var responses = await MassProcessAsync(hashes, (chunk) =>
            {
                var urlParams = new MassEditParams
                {
                    Urls = [data]
                };

                var request = new Request(Edit.Resource)
                {
                    Body = JObject.FromObject(urlParams, new JsonSerializer { NullValueHandling = NullValueHandling.Ignore })
                };

                request.Filter(Edit.Hashes, string.Join(',', chunk));

                return PatchAsync(request);
            });

            return new MassEditResponse
            {
                Urls = responses.SelectMany(r => r.GetList<MassEditResponse>(MassEditResponse.DataItem).Urls)
            };
        }
    }

    public async Task<DeleteResponse> DeleteAsync(string hash)
    {
        var request = new Request(UrlHelper.CombineUrl(Delete.Resource, hash));

        Response clientResponse = await DeleteAsync(request).ConfigureAwait(false);

        return clientResponse.GetValue<DeleteResponse>(DeleteResponse.DataItem);
    }

    public async Task<DeleteResponse> DeleteAsync(IEnumerable<string> hashes = null, IEnumerable<string> tags = null)
    {
        if (hashes == null && tags == null)
            throw new ArgumentException("Please specify Hashes or Tags selector to edit urls");
        
        if (hashes != null && tags != null)
            throw new ArgumentException("Hashes parameter doesn't work together with tags parameter");

        if (hashes == null)
        {
            // Simple API call
            var request = new Request(Delete.Resource);

            if (tags != null)
                request.Filter(Delete.Tags, string.Join(',', tags));

            Response clientResponse = await DeleteAsync(request).ConfigureAwait(false);

            return clientResponse.GetValue<DeleteResponse>(DeleteResponse.DataItem);
        }
        else
        {
            // Parallel API calls
            var responses = await MassProcessAsync(hashes, (chunk) =>
            {
                var request = new Request(Delete.Resource);

                request.Filter(Delete.Hashes, string.Join(',', chunk));

                return DeleteAsync(request);
            });

            return new DeleteResponse();
        }
    }

    public async Task<StatsResponse> StatsAsync(string hash)
    {
        var request = new Request(UrlHelper.CombineUrl(Stats.Resource, hash));

        Response clientResponse = await GetAsync(request).ConfigureAwait(false);

        return clientResponse.GetList<StatsResponse>(StatsResponse.DataItem);
    }

    public async Task<ResetStatsResponse> ResetStatsAsync(string hash)
    {
        var request = new Request(UrlHelper.CombineUrl(ResetStats.Resource, hash));

        Response clientResponse = await DeleteAsync(request).ConfigureAwait(false);

        return clientResponse.GetValue<ResetStatsResponse>(ResetStatsResponse.DataItem);
    }

    public async Task<ResetStatsResponse> ResetStatsAsync(IEnumerable<string> hashes = null, IEnumerable<string> tags = null)
    {
        if (hashes == null && tags == null)
            throw new ArgumentException("Please specify Hashes or Tags selector to edit urls");

        if (hashes != null && tags != null)
            throw new ArgumentException("Hashes parameter doesn't work together with tags parameter");

        if (hashes == null)
        {
            // Simple API call
            var request = new Request(ResetStats.Resource);

            if (tags != null)
                request.Filter(Delete.Tags, string.Join(',', tags));

            Response clientResponse = await DeleteAsync(request).ConfigureAwait(false);

            return clientResponse.GetValue<ResetStatsResponse>(ResetStatsResponse.DataItem);
        }
        else
        {
            // Parallel API calls
            var responses = await MassProcessAsync(hashes, (chunk) =>
            {
                var request = new Request(ResetStats.Resource);

                request.Filter(Delete.Hashes, string.Join(',', chunk));

                return DeleteAsync(request);
            });

            return new ResetStatsResponse();
        }
    }

    public async Task<MetaCriterionResponse> MetaCriterionAsync()
    {
        var request = new Request(MetaCriterion.Resource);

        Response clientResponse = await GetAsync(request).ConfigureAwait(false);

        return clientResponse.GetValue<MetaCriterionResponse>(MetaCriterionResponse.DataItem);
    }

    public async Task<Dictionary<string, MetaBrowserResponse>> MetaBrowserAsync()
    {
        var request = new Request(MetaBrowser.Resource);

        Response clientResponse = await GetAsync(request).ConfigureAwait(false);

        return clientResponse.GetDictionary<MetaBrowserResponse>(MetaBrowserResponse.DataItem);
    }

    public async Task<Dictionary<string, MetaCountryResponse>> MetaCountryAsync()
    {
        var request = new Request(MetaCountry.Resource);

        Response clientResponse = await GetAsync(request).ConfigureAwait(false);

        return clientResponse.GetDictionary<MetaCountryResponse>(MetaCountryResponse.DataItem);
    }

    public async Task<Dictionary<string, MetaLanguageResponse>> MetaLanguageAsync()
    {
        var request = new Request(MetaLanguage.Resource);

        Response clientResponse = await GetAsync(request).ConfigureAwait(false);

        return clientResponse.GetDictionary<MetaLanguageResponse>(MetaLanguageResponse.DataItem);
    }

    public async Task<Dictionary<string, MetaPlatformResponse>> MetaPlatformAsync()
    {
        var request = new Request(MetaPlatform.Resource);

        Response clientResponse = await GetAsync(request).ConfigureAwait(false);

        return clientResponse.GetDictionary<MetaPlatformResponse>(MetaPlatformResponse.DataItem);
    }

    public async Task<SmartLinkResponse> SmartLinkAsync(string hash)
    {
        var request = new Request(UrlHelper.CombineUrl(SmartLink.Resource, hash));

        Response clientResponse = await GetAsync(request).ConfigureAwait(false);

        return clientResponse.GetList<SmartLinkResponse>(SmartLinkResponse.DataItem);
    }

    public async Task<EditSmartLinkResponse> EditSmartLinkAsync(string hash, IEnumerable<Rule> rules)
    {
        var editParams = new EditSmartLinkParams
        {
            Rules = rules
        };

        var request = new Request(UrlHelper.CombineUrl(EditSmartLink.Resource, hash))
        {
            Body = JObject.FromObject(editParams, new JsonSerializer 
            {
                NullValueHandling = NullValueHandling.Ignore,
                Converters = { new StringEnumConverter(new CamelCaseNamingStrategy()) } 
            })
        };

        Response clientResponse = await PostAsync(request).ConfigureAwait(false);

        return clientResponse.GetList<EditSmartLinkResponse>(EditSmartLinkResponse.DataItem);
    }
}
