﻿using Newtonsoft.Json;
using Newtonsoft.Json.Linq;
using System.Text;
using Tinycc.Helpers;

namespace Tinycc;

public class BaseClient
{
    public string Domain { get; set; }

    protected int BatchOperationsLimit;

    protected int ParallelStreams;

    private HttpClient _httpClient;

    public BaseClient(string username, string apiKey, bool useBasicAuth, int batchOperationsLimit, int parallelStreams)
    {
        if (String.IsNullOrEmpty(username))
            throw new ArgumentException("username is required");

        if (String.IsNullOrEmpty(apiKey))
            throw new ArgumentException("apiKey is required");

        _httpClient = new HttpClient();
        _httpClient.SetDefaultSettings();

        if (useBasicAuth)
            _httpClient.UseBasicAuthentication(username, apiKey);
        else
            _httpClient.UseHeadersAuthentication(username, apiKey);

        BatchOperationsLimit = batchOperationsLimit;
        ParallelStreams = parallelStreams;
    }

    protected async Task<Response> GetAsync(Request request)
    {
        DomainFilter(request);

        string url = request.BuildUrl();

        var responseMessage = await _httpClient.GetAsync(url).ConfigureAwait(false);

        JObject content = await HttpContentHelper.GetContentAsync(responseMessage).ConfigureAwait(false);
        return new Response(responseMessage.IsSuccessStatusCode, (int)responseMessage.StatusCode, content);
    }

    protected async Task<Response> PostAsync(Request request)
    {
        DomainFilter(request);

        string url = request.BuildUrl();

        var output = request.Body.ToString(Formatting.None);
        HttpContent contentPost = new StringContent(output, Encoding.UTF8, Constants.JsonMediaType);
        var responseMessage = await _httpClient.PostAsync(url, contentPost).ConfigureAwait(false);

        JObject content = await HttpContentHelper.GetContentAsync(responseMessage).ConfigureAwait(false);
        return new Response(responseMessage.IsSuccessStatusCode, (int)responseMessage.StatusCode, content);
    }

    protected async Task<Response> PutAsync(Request request)
    {
        DomainFilter(request);

        string url = request.BuildUrl();

        var output = request.Body.ToString(Formatting.None);
        HttpContent contentPut = new StringContent(output, Encoding.UTF8, Constants.JsonMediaType);
        var responseMessage = await _httpClient.PutAsync(url, contentPut).ConfigureAwait(false);

        JObject content = await HttpContentHelper.GetContentAsync(responseMessage).ConfigureAwait(false);
        Response mailjetResponse = new Response(responseMessage.IsSuccessStatusCode, (int)responseMessage.StatusCode, content);
        return mailjetResponse;
    }

    protected async Task<Response> PatchAsync(Request request)
    {
        DomainFilter(request);

        string url = request.BuildUrl();

        var output = request.Body.ToString(Formatting.None);
        HttpContent contentPost = new StringContent(output, Encoding.UTF8, Constants.JsonMediaType);
        var responseMessage = await _httpClient.PatchAsync(url, contentPost).ConfigureAwait(false);

        JObject content = await HttpContentHelper.GetContentAsync(responseMessage).ConfigureAwait(false);
        return new Response(responseMessage.IsSuccessStatusCode, (int)responseMessage.StatusCode, content);
    }

    protected async Task<Response> DeleteAsync(Request request)
    {
        DomainFilter(request);

        string url = request.BuildUrl();

        var responseMessage = await _httpClient.DeleteAsync(url).ConfigureAwait(false);

        JObject content = await HttpContentHelper.GetContentAsync(responseMessage).ConfigureAwait(false);
        return new Response(responseMessage.IsSuccessStatusCode, (int)responseMessage.StatusCode, content);
    }

    protected async Task<IEnumerable<Response>> MassProcessAsync<T>(IEnumerable<T> list, Func<IEnumerable<T>, Task<Response>> startRequest)
    {
        List<Response> responses = [];
        List<Task<Response>> streams = [];
        foreach (var chunk in list.Chunk(BatchOperationsLimit))
        {
            streams.Add(startRequest(chunk));
            if (streams.Count >= ParallelStreams)
            {
                await Task.WhenAll(streams).ConfigureAwait(false);
                responses.AddRange(streams.Select(task => task.Result));
                streams.Clear();
            }
        }
        await Task.WhenAll(streams).ConfigureAwait(false);
        responses.AddRange(streams.Select(task => task.Result));
        return responses;
    }

    private void DomainFilter(Request request)
    {
        if (!String.IsNullOrEmpty(Domain))
            request.Filter("domain", Domain);
    }
}
